<?php
/*
 * Copyright (c) 2025 AltumCode (https://altumcode.com/)
 *
 * This software is licensed exclusively by AltumCode and is sold only via https://altumcode.com/.
 * Unauthorized distribution, modification, or use of this software without a valid license is not permitted and may be subject to applicable legal actions.
 *
 * 🌍 View all other existing AltumCode projects via https://altumcode.com/
 * 📧 Get in touch for support or general queries via https://altumcode.com/contact
 * 📤 Download the latest version via https://altumcode.com/downloads
 *
 * 🐦 X/Twitter: https://x.com/AltumCode
 * 📘 Facebook: https://facebook.com/altumcode
 * 📸 Instagram: https://instagram.com/altumcode
 */

namespace Altum\Controllers;

use Altum\Alerts;
use Altum\Models\PersonalNotification;

defined('ALTUMCODE') || die();

class AdminPersonalNotifications extends Controller {

    public function index() {

        /* Prepare the filtering system */
        $filters = (new \Altum\Filters(['website_id', 'type', 'user_id'], ['name', 'title', 'description', 'url',], ['personal_notification_id', 'name', 'title', 'sent_datetime', 'datetime', 'last_datetime', ]));
        $filters->set_default_order_by($this->user->preferences->personal_notifications_default_order_by, $this->user->preferences->default_order_type ?? settings()->main->default_order_type);
        $filters->set_default_results_per_page($this->user->preferences->default_results_per_page ?? settings()->main->default_results_per_page);

        /* Prepare the paginator */
        $total_rows = database()->query("SELECT COUNT(*) AS `total` FROM `personal_notifications` WHERE 1 = 1 {$filters->get_sql_where()}")->fetch_object()->total ?? 0;
        $paginator = (new \Altum\Paginator($total_rows, $filters->get_results_per_page(), $_GET['page'] ?? 1, url('admin/personal-notifications?' . $filters->get_get() . '&page=%d')));

        /* Get the campaigns list for the user */
        $personal_notifications = [];
        $personal_notifications_result = database()->query("
            SELECT
                `personal_notifications`.*, `users`.`name` AS `user_name`, `users`.`email` AS `user_email`, `users`.`avatar` AS `user_avatar`, `websites`.`host`, `websites`.`path`, `subscribers`.`ip`
            FROM
                `personal_notifications`
            LEFT JOIN
                `users` ON `personal_notifications`.`user_id` = `users`.`user_id`
            LEFT JOIN
                `websites` ON `personal_notifications`.`website_id` = `websites`.`website_id`
            LEFT JOIN
                `subscribers` ON `personal_notifications`.`subscriber_id` = `subscribers`.`subscriber_id`
            WHERE
                1 = 1
                {$filters->get_sql_where('personal_notifications')}
                {$filters->get_sql_order_by('personal_notifications')}
            
            {$paginator->get_sql_limit()}
        ");
        while($row = $personal_notifications_result->fetch_object()) {
            $row->settings = json_decode($row->settings ?? '');
            $personal_notifications[] = $row;
        }

        /* Export handler */
        process_export_json($personal_notifications, ['personal_notification_id', 'website_id', 'user_id', 'name', 'title', 'description', 'url', 'image', 'settings', 'status', 'is_sent', 'is_displayed', 'is_clicked', 'is_closed', 'scheduled_datetime', 'sent_datetime', 'datetime', 'last_datetime',]);
        process_export_csv_new($personal_notifications, ['personal_notification_id', 'website_id', 'user_id', 'name', 'title', 'description', 'url', 'image', 'settings', 'status', 'is_sent', 'is_displayed', 'is_clicked', 'is_closed', 'scheduled_datetime', 'sent_datetime', 'datetime', 'last_datetime',], ['settings']);

        /* Prepare the pagination view */
        $pagination = (new \Altum\View('partials/pagination', (array) $this))->run(['paginator' => $paginator]);

        /* Prepare the view */
        $data = [
            'personal_notifications' => $personal_notifications,
            'pagination' => $pagination,
            'filters' => $filters,
        ];

        $view = new \Altum\View('admin/personal-notifications/index', (array) $this);

        $this->add_view_content('content', $view->run($data));

    }

    public function bulk() {

        /* Check for any errors */
        if(empty($_POST)) {
            redirect('admin/personal-notifications');
        }

        if(empty($_POST['selected'])) {
            redirect('admin/personal-notifications');
        }

        if(!isset($_POST['type'])) {
            redirect('admin/personal-notifications');
        }

        //ALTUMCODE:DEMO if(DEMO) Alerts::add_error('This command is blocked on the demo.');

        if(!\Altum\Csrf::check()) {
            Alerts::add_error(l('global.error_message.invalid_csrf_token'));
        }

        if(!Alerts::has_field_errors() && !Alerts::has_errors()) {

            set_time_limit(0);

            session_write_close();

            switch($_POST['type']) {
                case 'delete':

                    foreach($_POST['selected'] as $personal_notification_id) {
                        (new PersonalNotification())->delete($personal_notification_id);
                    }

                    break;
            }

            session_start();

            /* Set a nice success message */
            Alerts::add_success(l('bulk_delete_modal.success_message'));

        }

        redirect('admin/personal-notifications');
    }

    public function delete() {

        $personal_notification_id = (isset($this->params[0])) ? (int) $this->params[0] : null;

        //ALTUMCODE:DEMO if(DEMO) Alerts::add_error('This command is blocked on the demo.');

        if(!\Altum\Csrf::check('global_token')) {
            Alerts::add_error(l('global.error_message.invalid_csrf_token'));
        }

        if(!$personal_notification = db()->where('personal_notification_id', $personal_notification_id)->getOne('personal_notifications', ['personal_notification_id', 'name'])) {
            redirect('admin/personal-notifications');
        }

        if(!Alerts::has_field_errors() && !Alerts::has_errors()) {

            (new PersonalNotification())->delete($personal_notification_id);

            /* Set a nice success message */
            Alerts::add_success(sprintf(l('global.success_message.delete1'), '<strong>' . $personal_notification->name . '</strong>'));

        }

        redirect('admin/personal-notifications');
    }
}
